#!/usr/bin/env python3
"""
Channel Sounding Proximity-Based Mac Lock/Unlock Demo
=====================================================

Reads distance measurements from a Silicon Labs Channel Sounding dev kit
via serial port and locks/unlocks the Mac based on proximity.

Requirements:
    pip install pyserial keyring pyobjc-framework-Quartz

Setup:
    1. Grant Accessibility permissions to Terminal/your IDE
    2. Run: python cs_proximity_unlock.py --setup  (to store password securely)
    3. Run: python cs_proximity_unlock.py /dev/cu.usbmodem*  (to start)

Author: Mohammad Afaneh / Novel Bits
For educational/demo purposes - Channel Sounding technology demonstration
"""

import argparse
import re
import subprocess
import sys
import termios
import time
from collections import deque
from dataclasses import dataclass
from enum import Enum, auto
from typing import Optional

import serial
import keyring

# macOS-specific imports
try:
    import Quartz
    from Quartz import (
        CGEventCreateKeyboardEvent,
        CGEventPost,
        CGEventSetFlags,
        kCGHIDEventTap,
        kCGEventFlagMaskShift,
    )
    # For unicode string input
    import Quartz.CoreGraphics as CG
    MACOS_AVAILABLE = True
except ImportError:
    MACOS_AVAILABLE = False
    print("Warning: Quartz not available. Lock/unlock will use fallback methods.")


# =============================================================================
# Configuration
# =============================================================================

KEYRING_SERVICE = "cs_proximity_unlock"
KEYRING_USERNAME = "mac_password"

# Distance thresholds (in millimeters)
UNLOCK_DISTANCE_MM = 800      # Unlock when closer than this
LOCK_DISTANCE_MM = 2000       # Lock when farther than this

# Likelihood threshold - only trust measurements with likelihood above this
MIN_LIKELIHOOD = 0.45

# Hysteresis settings - require N consecutive readings before state change
UNLOCK_CONSECUTIVE_READINGS = 3
LOCK_CONSECUTIVE_READINGS = 5

# Smoothing - use moving average of last N readings
SMOOTHING_WINDOW = 5


# =============================================================================
# UI Styling
# =============================================================================

class Colors:
    """ANSI color codes for terminal output."""
    # Basic colors
    RESET = '\033[0m'
    BOLD = '\033[1m'
    DIM = '\033[2m'

    # Foreground colors
    RED = '\033[91m'
    GREEN = '\033[92m'
    YELLOW = '\033[93m'
    BLUE = '\033[94m'
    MAGENTA = '\033[95m'
    CYAN = '\033[96m'
    WHITE = '\033[97m'
    GRAY = '\033[90m'

    # Background colors
    BG_RED = '\033[101m'
    BG_GREEN = '\033[102m'
    BG_YELLOW = '\033[103m'
    BG_BLUE = '\033[104m'

    @classmethod
    def disable(cls):
        """Disable colors (for non-TTY output)."""
        for attr in dir(cls):
            if attr.isupper() and not attr.startswith('_'):
                setattr(cls, attr, '')


class UI:
    """User interface helper for elegant terminal output."""

    # Check if we're in a TTY (interactive terminal)
    IS_TTY = sys.stdout.isatty()

    # Symbols
    ICON_LOCKED = '🔒'
    ICON_UNLOCKED = '🔓'
    ICON_DEVICE = '📱'
    ICON_CHECK = '✓'
    ICON_CROSS = '✗'
    ICON_WARN = '⚠'
    ICON_DOT = '●'
    ICON_EMPTY = '○'
    ICON_BLUETOOTH = '◉'

    # Distance bar characters
    BAR_FULL = '█'
    BAR_EMPTY = '░'
    BAR_UNLOCK = '┃'
    BAR_LOCK = '┃'

    @classmethod
    def clear_line(cls):
        """Clear the current line."""
        if cls.IS_TTY:
            print('\r\033[K', end='')

    @classmethod
    def banner(cls, mode: str, port: str, unlock_dist: int, lock_dist: int):
        """Print an elegant startup banner."""
        c = Colors
        width = 52

        print()
        print(f"{c.CYAN}{c.BOLD}{'─' * width}{c.RESET}")
        print(f"{c.CYAN}{c.BOLD}  {cls.ICON_BLUETOOTH}  Channel Sounding Proximity Unlock{c.RESET}")
        print(f"{c.CYAN}{'─' * width}{c.RESET}")
        print()
        print(f"  {c.DIM}Mode:{c.RESET}      {c.WHITE}{mode}{c.RESET}")
        print(f"  {c.DIM}Port:{c.RESET}      {c.WHITE}{port}{c.RESET}")
        print(f"  {c.DIM}Unlock:{c.RESET}    {c.GREEN}< {unlock_dist/1000:.1f}m{c.RESET}")
        print(f"  {c.DIM}Lock:{c.RESET}      {c.RED}> {lock_dist/1000:.1f}m{c.RESET}")
        print()
        print(f"{c.CYAN}{'─' * width}{c.RESET}")
        print(f"  {c.DIM}Waiting for device...{c.RESET}")
        print()

    @classmethod
    def device_connected(cls, address: str):
        """Show device connected message."""
        c = Colors
        cls.clear_line()
        print(f"\n  {c.GREEN}{c.BOLD}{cls.ICON_DEVICE} Connected:{c.RESET} {c.WHITE}{address}{c.RESET}\n")

    @classmethod
    def device_disconnected(cls):
        """Show device disconnected message."""
        c = Colors
        cls.clear_line()
        print(f"\n  {c.YELLOW}{cls.ICON_DEVICE} Disconnected{c.RESET}\n")

    @classmethod
    def soc_reset(cls):
        """Show SOC reset message."""
        c = Colors
        cls.clear_line()
        print(f"\n  {c.MAGENTA}{cls.ICON_BLUETOOTH} SOC Reset detected - waiting for reconnection...{c.RESET}\n")

    @classmethod
    def lock_event(cls, success: bool):
        """Show lock event."""
        c = Colors
        cls.clear_line()
        if success:
            print(f"\n  {c.RED}{c.BOLD}{cls.ICON_LOCKED} Screen Locked{c.RESET}\n")
        else:
            print(f"\n  {c.RED}{cls.ICON_CROSS} Lock Failed{c.RESET}\n")

    @classmethod
    def unlock_event(cls, success: bool):
        """Show unlock event."""
        c = Colors
        cls.clear_line()
        if success:
            print(f"\n  {c.GREEN}{c.BOLD}{cls.ICON_UNLOCKED} Screen Unlocked{c.RESET}\n")
        else:
            print(f"\n  {c.YELLOW}{cls.ICON_CROSS} Unlock Failed{c.RESET}\n")

    @classmethod
    def distance_bar(cls, distance_mm: int, smoothed_mm: float, likelihood: float,
                     unlock_threshold: int, lock_threshold: int, max_dist: int = 3000):
        """Render an elegant distance bar."""
        c = Colors

        # Calculate bar position
        bar_width = 30
        pos = min(distance_mm, max_dist) / max_dist
        fill_count = int(pos * bar_width)

        # Threshold positions
        unlock_pos = int(unlock_threshold / max_dist * bar_width)
        lock_pos = int(lock_threshold / max_dist * bar_width)

        # Build the bar with color zones
        bar = ''
        for i in range(bar_width):
            if i < fill_count:
                if i < unlock_pos:
                    bar += f"{c.GREEN}{cls.BAR_FULL}"
                elif i < lock_pos:
                    bar += f"{c.YELLOW}{cls.BAR_FULL}"
                else:
                    bar += f"{c.RED}{cls.BAR_FULL}"
            else:
                bar += f"{c.GRAY}{cls.BAR_EMPTY}"
        bar += c.RESET

        # Confidence indicator
        if likelihood >= 0.7:
            conf = f"{c.GREEN}{cls.ICON_DOT}{cls.ICON_DOT}{cls.ICON_DOT}{c.RESET}"
            conf_label = "High"
        elif likelihood >= 0.55:
            conf = f"{c.YELLOW}{cls.ICON_DOT}{cls.ICON_DOT}{cls.ICON_EMPTY}{c.RESET}"
            conf_label = "Med"
        elif likelihood >= MIN_LIKELIHOOD:
            conf = f"{c.YELLOW}{cls.ICON_DOT}{cls.ICON_EMPTY}{cls.ICON_EMPTY}{c.RESET}"
            conf_label = "Low"
        else:
            conf = f"{c.RED}{cls.ICON_EMPTY}{cls.ICON_EMPTY}{cls.ICON_EMPTY}{c.RESET}"
            conf_label = "Poor"

        # Distance display with color
        if distance_mm < unlock_threshold:
            dist_color = c.GREEN
            zone = "NEAR"
        elif distance_mm > lock_threshold:
            dist_color = c.RED
            zone = "FAR"
        else:
            dist_color = c.YELLOW
            zone = "MID"

        # Format the display
        smoothed_str = f"{smoothed_mm/1000:.2f}" if smoothed_mm else "-.--"
        dist_m = distance_mm / 1000

        status = (
            f"\r  {dist_color}{c.BOLD}{dist_m:5.2f}m{c.RESET} "
            f"{c.DIM}(avg: {smoothed_str}m){c.RESET}  "
            f"[{bar}]  "
            f"{conf} "
        )

        print(status, end='', flush=True)

    @classmethod
    def timeout_warning(cls):
        """Show timeout warning."""
        c = Colors
        cls.clear_line()
        print(f"\n  {c.YELLOW}{cls.ICON_WARN} Signal lost - device may be out of range{c.RESET}\n")

    @classmethod
    def safety_warning(cls):
        """Show safety check warning."""
        c = Colors
        print(f"  {c.YELLOW}{cls.ICON_WARN} Safety check: waiting for lock screen...{c.RESET}")

    @classmethod
    def info(cls, message: str):
        """Show info message."""
        c = Colors
        print(f"  {c.DIM}{message}{c.RESET}")

    @classmethod
    def error(cls, message: str):
        """Show error message."""
        c = Colors
        print(f"  {c.RED}{cls.ICON_CROSS} {message}{c.RESET}")

    @classmethod
    def success(cls, message: str):
        """Show success message."""
        c = Colors
        print(f"  {c.GREEN}{cls.ICON_CHECK} {message}{c.RESET}")


# Disable colors if not in a TTY
if not UI.IS_TTY:
    Colors.disable()


# =============================================================================
# Data Structures
# =============================================================================

@dataclass
class CSMeasurement:
    """A single Channel Sounding measurement."""
    bt_address: str
    distance_mm: int
    likelihood: float
    velocity_mps: float
    timestamp: float


class LockState(Enum):
    """Mac lock state."""
    UNKNOWN = auto()
    LOCKED = auto()
    UNLOCKED = auto()


# =============================================================================
# Serial Parser
# =============================================================================

class CSSerialParser:
    """Parses Channel Sounding output from Silicon Labs dev kit."""
    
    # Pattern for measurement lines:
    # [APP] [1] 55:B3:D3:F6:CD:2F |  1475 mm | 0.62  | -0.54 m/s
    MEASUREMENT_PATTERN = re.compile(
        r'\[APP\]\s+\[\d+\]\s+'
        r'([0-9A-Fa-f:]{17})\s*\|\s*'  # BT Address
        r'(\d+)\s*mm\s*\|\s*'           # Distance in mm
        r'([\d.]+)\s*\|\s*'             # Likelihood
        r'([+-]?[\d.]+)\s*m/s'          # Velocity
    )
    
    # Pattern for connection events
    CONNECTION_PATTERN = re.compile(
        r'\[APP\].*Connection opened.*with CS Reflector\s+\'([0-9A-Fa-f:]{17})\''
    )
    
    DISCONNECTION_PATTERN = re.compile(
        r'\[APP\].*(Connection closed|disconnected)',
        re.IGNORECASE
    )

    # Pattern for SOC reset (detects the Silicon Labs startup banner)
    RESET_PATTERN = re.compile(
        r'\+-\[CS initiator by Silicon Labs\]'
    )

    def parse_line(self, line: str) -> Optional[CSMeasurement]:
        """Parse a single line from serial output."""
        match = self.MEASUREMENT_PATTERN.search(line)
        if match:
            return CSMeasurement(
                bt_address=match.group(1),
                distance_mm=int(match.group(2)),
                likelihood=float(match.group(3)),
                velocity_mps=float(match.group(4)),
                timestamp=time.time()
            )
        return None
    
    def is_connection_event(self, line: str) -> Optional[str]:
        """Check if line indicates a new connection. Returns BT address if so."""
        match = self.CONNECTION_PATTERN.search(line)
        return match.group(1) if match else None
    
    def is_disconnection_event(self, line: str) -> bool:
        """Check if line indicates a disconnection."""
        return bool(self.DISCONNECTION_PATTERN.search(line))

    def is_reset_event(self, line: str) -> bool:
        """Check if line indicates SOC reset (startup banner detected)."""
        return bool(self.RESET_PATTERN.search(line))


# =============================================================================
# Distance Processor with Smoothing and Hysteresis
# =============================================================================

class DistanceProcessor:
    """Processes distance measurements with smoothing and hysteresis."""
    
    def __init__(
        self,
        unlock_distance: int = UNLOCK_DISTANCE_MM,
        lock_distance: int = LOCK_DISTANCE_MM,
        min_likelihood: float = MIN_LIKELIHOOD,
        unlock_consecutive: int = UNLOCK_CONSECUTIVE_READINGS,
        lock_consecutive: int = LOCK_CONSECUTIVE_READINGS,
        smoothing_window: int = SMOOTHING_WINDOW
    ):
        self.unlock_distance = unlock_distance
        self.lock_distance = lock_distance
        self.min_likelihood = min_likelihood
        self.unlock_consecutive = unlock_consecutive
        self.lock_consecutive = lock_consecutive
        
        # Rolling window for smoothing
        self.distance_history = deque(maxlen=smoothing_window)
        
        # Consecutive counter for hysteresis
        self.unlock_counter = 0
        self.lock_counter = 0
        
        # Track last valid measurement time
        self.last_measurement_time = 0
        self.measurement_timeout = 3.0  # seconds
    
    def add_measurement(self, measurement: CSMeasurement) -> Optional[str]:
        """
        Add a measurement and return action if threshold crossed.
        
        Returns:
            'unlock' - if should unlock
            'lock' - if should lock
            None - no action needed
        """
        # Note: Likelihood filtering removed per Silicon Labs recommendation
        # All measurements are now used regardless of likelihood score

        self.last_measurement_time = measurement.timestamp
        self.distance_history.append(measurement.distance_mm)
        
        # Calculate smoothed distance
        if len(self.distance_history) < 2:
            return None
        
        smoothed_distance = sum(self.distance_history) / len(self.distance_history)
        
        # Check unlock condition
        if smoothed_distance < self.unlock_distance:
            self.unlock_counter += 1
            self.lock_counter = 0
            
            if self.unlock_counter >= self.unlock_consecutive:
                self.unlock_counter = 0
                return 'unlock'
        
        # Check lock condition
        elif smoothed_distance > self.lock_distance:
            self.lock_counter += 1
            self.unlock_counter = 0
            
            if self.lock_counter >= self.lock_consecutive:
                self.lock_counter = 0
                return 'lock'
        
        else:
            # In the dead zone between thresholds - reset counters slowly
            self.unlock_counter = max(0, self.unlock_counter - 1)
            self.lock_counter = max(0, self.lock_counter - 1)
        
        return None
    
    def check_timeout(self) -> bool:
        """Check if measurements have timed out (device out of range)."""
        if self.last_measurement_time == 0:
            return False
        return (time.time() - self.last_measurement_time) > self.measurement_timeout
    
    def get_smoothed_distance(self) -> Optional[float]:
        """Get current smoothed distance."""
        if not self.distance_history:
            return None
        return sum(self.distance_history) / len(self.distance_history)
    
    def reset(self):
        """Reset all state."""
        self.distance_history.clear()
        self.unlock_counter = 0
        self.lock_counter = 0
        self.last_measurement_time = 0


# =============================================================================
# macOS Lock/Unlock Controller
# =============================================================================

# Virtual key codes for common keys (from Events.h)
KEY_CODES = {
    'return': 36,
    'tab': 48,
    'space': 49,
    'delete': 51,
    'escape': 53,
    'command': 55,
    'shift': 56,
    'capslock': 57,
    'option': 58,
    'control': 59,
}


class MacOSController:
    """Controls macOS lock screen state using native Quartz APIs."""
    
    def __init__(self):
        self.current_state = LockState.UNKNOWN
        self._password: Optional[str] = None
    
    def load_password(self) -> bool:
        """Load password from keychain."""
        try:
            self._password = keyring.get_password(KEYRING_SERVICE, KEYRING_USERNAME)
            return self._password is not None
        except Exception as e:
            UI.error(f"Loading password: {e}")
            return False
    
    @staticmethod
    def store_password(password: str) -> bool:
        """Store password in keychain."""
        try:
            keyring.set_password(KEYRING_SERVICE, KEYRING_USERNAME, password)
            return True
        except Exception as e:
            UI.error(f"Storing password: {e}")
            return False
    
    @staticmethod
    def delete_password() -> bool:
        """Delete password from keychain."""
        try:
            keyring.delete_password(KEYRING_SERVICE, KEYRING_USERNAME)
            UI.success("Password deleted from keychain")
            return True
        except keyring.errors.PasswordDeleteError:
            UI.info("No password found in keychain")
            return False
    
    def is_screen_locked(self) -> bool:
        """Check if screen is currently locked."""
        if MACOS_AVAILABLE:
            # Check if screen saver or login window is active
            session_dict = Quartz.CGSessionCopyCurrentDictionary()
            if session_dict:
                # CGSSessionScreenIsLocked key indicates lock state
                is_locked = session_dict.get('CGSSessionScreenIsLocked', False)
                return bool(is_locked)

        # Fallback: check if loginwindow is frontmost
        return self._is_loginwindow_frontmost()

    def _is_loginwindow_frontmost(self) -> bool:
        """Check if loginwindow is the frontmost application."""
        try:
            result = subprocess.run(
                ['osascript', '-e',
                 'tell application "System Events" to get name of first process whose frontmost is true'],
                capture_output=True, text=True, timeout=2
            )
            return 'loginwindow' in result.stdout.lower()
        except Exception:
            return False

    def _is_unsafe_app_frontmost(self) -> bool:
        """
        Check if an unsafe application (Terminal, iTerm, etc.) is frontmost.
        SECURITY: If any of these apps are frontmost, we should NOT type the password.
        """
        unsafe_apps = ['terminal', 'iterm', 'hyper', 'alacritty', 'kitty', 'warp']
        try:
            result = subprocess.run(
                ['osascript', '-e',
                 'tell application "System Events" to get name of first process whose frontmost is true'],
                capture_output=True, text=True, timeout=2
            )
            frontmost = result.stdout.strip().lower()
            return any(app in frontmost for app in unsafe_apps)
        except Exception:
            return True  # If we can't check, assume unsafe

    def _is_safe_to_type_password(self) -> bool:
        """
        Safety check: Return True if we believe the lock screen is active.
        This prevents typing the password into the wrong application.

        Uses OR logic: Quartz lock status OR loginwindow frontmost OR no unsafe app.
        """
        # Check 1: Quartz session says screen is locked (with retries)
        # When this is True, CGEvent keystrokes go to the lock screen
        if MACOS_AVAILABLE:
            for _ in range(5):
                session_dict = Quartz.CGSessionCopyCurrentDictionary()
                if session_dict:
                    if bool(session_dict.get('CGSSessionScreenIsLocked', False)):
                        return True
                time.sleep(0.1)

        # Check 2: loginwindow is frontmost (fallback with retry)
        for _ in range(3):
            if self._is_loginwindow_frontmost():
                return True
            time.sleep(0.1)

        # Check 3: Last resort - if no dangerous app is frontmost, proceed with caution
        # This handles cases where macOS doesn't report lock status correctly
        if not self._is_unsafe_app_frontmost():
            return True

        # Definitely unsafe - a terminal or similar app is frontmost
        return False
    
    def lock_screen(self) -> bool:
        """Lock the Mac screen."""
        
        try:
            # Method 1: Use CGSession (most reliable)
            subprocess.run(
                ['/System/Library/CoreServices/Menu Extras/User.menu/Contents/Resources/CGSession', '-suspend'],
                check=True, timeout=5
            )
            self.current_state = LockState.LOCKED
            return True
        except (subprocess.CalledProcessError, FileNotFoundError):
            pass
        
        try:
            # Method 2: pmset display sleep
            subprocess.run(['pmset', 'displaysleepnow'], check=True, timeout=5)
            self.current_state = LockState.LOCKED
            return True
        except subprocess.CalledProcessError:
            pass
        
        try:
            # Method 3: AppleScript
            subprocess.run([
                'osascript', '-e',
                'tell application "System Events" to keystroke "q" using {command down, control down}'
            ], check=True, timeout=5)
            self.current_state = LockState.LOCKED
            return True
        except subprocess.CalledProcessError:
            return False
    
    def _wake_display(self) -> bool:
        """Wake the display from sleep."""
        try:
            # Use caffeinate to wake display (1s is enough to trigger wake)
            subprocess.run(['caffeinate', '-u', '-t', '1'], timeout=3)
            return True
        except Exception as e:
            print(f"Failed to wake display: {e}")
            return False

    def _suppress_terminal_echo(self) -> Optional[list]:
        """
        Suppress terminal echo to prevent password leakage.
        Returns the original terminal settings for restoration.
        """
        try:
            old_settings = termios.tcgetattr(sys.stdin)
            new_settings = termios.tcgetattr(sys.stdin)
            # Disable echo (ECHO) and canonical mode (ICANON)
            new_settings[3] = new_settings[3] & ~termios.ECHO & ~termios.ICANON
            termios.tcsetattr(sys.stdin, termios.TCSADRAIN, new_settings)
            return old_settings
        except Exception:
            return None

    def _restore_terminal_echo(self, old_settings: Optional[list]) -> None:
        """Restore terminal settings after password typing."""
        if old_settings is not None:
            try:
                termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_settings)
            except Exception:
                pass

    def _type_string_quartz(self, text: str) -> bool:
        """
        Type a string using Quartz CGEvent APIs.
        Types character by character for reliable lock screen targeting.
        """
        if not MACOS_AVAILABLE:
            return False

        try:
            for char in text:
                # Create key down event
                key_down = CGEventCreateKeyboardEvent(None, 0, True)
                if key_down is None:
                    print("Failed to create key down event")
                    return False

                # Set the unicode character
                CG.CGEventKeyboardSetUnicodeString(key_down, len(char), char)

                # Post key down
                CGEventPost(kCGHIDEventTap, key_down)

                # Create key up event
                key_up = CGEventCreateKeyboardEvent(None, 0, False)
                CG.CGEventKeyboardSetUnicodeString(key_up, len(char), char)

                # Post key up
                CGEventPost(kCGHIDEventTap, key_up)

                # Delay between characters for reliable lock screen targeting
                time.sleep(0.01)

            return True

        except Exception as e:
            print(f"Quartz typing error: {e}")
            return False
    
    def _press_return_quartz(self) -> bool:
        """Press the Return key using Quartz."""
        if not MACOS_AVAILABLE:
            return False

        try:
            # Key code 36 = Return
            key_down = CGEventCreateKeyboardEvent(None, 36, True)
            CGEventPost(kCGHIDEventTap, key_down)

            key_up = CGEventCreateKeyboardEvent(None, 36, False)
            CGEventPost(kCGHIDEventTap, key_up)

            return True
        except Exception as e:
            print(f"Failed to press Return: {e}")
            return False

    def _press_escape_quartz(self) -> bool:
        """
        Press Escape key to abort/clear any partial password input.
        SECURITY: Called when safety check fails mid-typing to clear the password field.
        """
        if not MACOS_AVAILABLE:
            return False

        try:
            # Key code 53 = Escape
            key_down = CGEventCreateKeyboardEvent(None, 53, True)
            CGEventPost(kCGHIDEventTap, key_down)

            key_up = CGEventCreateKeyboardEvent(None, 53, False)
            CGEventPost(kCGHIDEventTap, key_up)

            return True
        except Exception:
            return False

    def _clear_password_field(self) -> None:
        """
        Emergency: Select all and delete to clear any partial password.
        SECURITY: Called when aborting to ensure no password characters remain.
        """
        if not MACOS_AVAILABLE:
            return

        try:
            # Press Escape first
            self._press_escape_quartz()
            time.sleep(0.05)

            # Cmd+A (select all) - key code 0 = 'a', modifier = command
            cmd_a_down = CGEventCreateKeyboardEvent(None, 0, True)
            CG.CGEventSetFlags(cmd_a_down, CG.kCGEventFlagMaskCommand)
            CGEventPost(kCGHIDEventTap, cmd_a_down)

            cmd_a_up = CGEventCreateKeyboardEvent(None, 0, False)
            CGEventPost(kCGHIDEventTap, cmd_a_up)

            time.sleep(0.02)

            # Delete (backspace) - key code 51
            delete_down = CGEventCreateKeyboardEvent(None, 51, True)
            CGEventPost(kCGHIDEventTap, delete_down)

            delete_up = CGEventCreateKeyboardEvent(None, 51, False)
            CGEventPost(kCGHIDEventTap, delete_up)

        except Exception:
            pass  # Best effort - don't fail on cleanup

    def _verify_unlock_success(self, timeout: float = 0.3) -> bool:
        """Poll for unlock success instead of fixed wait."""
        elapsed = 0.0
        while elapsed < timeout:
            if not self.is_screen_locked():
                return True
            time.sleep(0.05)
            elapsed += 0.05
        return False

    def _unlock_with_quartz(self) -> bool:
        """
        Unlock using direct Quartz keyboard injection.
        This mimics exactly what BLEUnlock does.

        SECURITY: Verifies loginwindow is frontmost before typing password.
        """
        if not self._password:
            return False

        # Wake the display first
        self._wake_display()

        # Wait for lock screen to fully render
        time.sleep(1.0)

        # SAFETY CHECK: Verify lock screen is active before typing password
        # If this fails, silently return False to allow AppleScript fallback
        if not self._is_safe_to_type_password():
            return False

        # Suppress terminal echo to prevent password leakage to terminal
        old_term = self._suppress_terminal_echo()

        try:
            # Type the password character by character
            if not self._type_string_quartz(self._password):
                return False

            time.sleep(0.03)  # Brief settle time

            # Press Return to submit
            if not self._press_return_quartz():
                print("Failed to press Return")
                return False

            # Brief delay to let unlock begin before returning control
            time.sleep(0.05)

            return True
        finally:
            # Always restore terminal settings
            self._restore_terminal_echo(old_term)
    
    def _unlock_with_applescript(self, skip_wake: bool = False) -> bool:
        """
        Fallback: unlock using AppleScript (slower but more compatible).

        SECURITY: Verifies loginwindow is frontmost before typing password.
        Note: This method is less secure than Quartz as password appears briefly
        in process list. Prefer Quartz method when available.
        """
        if not self._password:
            return False

        try:
            # Wake display only if not already done by Quartz attempt
            if not skip_wake:
                subprocess.run(['caffeinate', '-u', '-t', '1'], timeout=3)
                # Wait for lock screen to fully render
                time.sleep(1.0)
            else:
                # Called as fallback - give system a moment to stabilize
                time.sleep(0.3)

            # SAFETY CHECK: Verify login window is active before typing password
            # (has built-in retries for timing during wake)
            if not self._is_safe_to_type_password():
                UI.safety_warning()
                return False

            # Escape any special characters for AppleScript
            escaped_password = self._password.replace('\\', '\\\\').replace('"', '\\"')

            # Type password and press return
            script = f'''
            tell application "System Events"
                keystroke "{escaped_password}"
                delay 0.05
                keystroke return
            end tell
            '''

            result = subprocess.run(
                ['osascript', '-e', script],
                capture_output=True, text=True, timeout=10
            )

            return result.returncode == 0

        except Exception as e:
            # On any exception, try to clear password field
            self._clear_password_field()
            return False

    def unlock_screen(self) -> bool:
        """Unlock the Mac screen by entering password."""
        if not self._password:
            UI.error("No password configured")
            return False

        if not self.is_screen_locked():
            self.current_state = LockState.UNLOCKED
            return True

        # Try Quartz method first (faster, like BLEUnlock)
        if MACOS_AVAILABLE:
            success = self._unlock_with_quartz()
            if success:
                # Quartz submitted the password - give macOS enough time to unlock
                if self._verify_unlock_success(timeout=2.0):
                    self.current_state = LockState.UNLOCKED
                    return True
                # Password was already typed - do NOT fall through to AppleScript
                # or it will re-type the password into the now-unlocked terminal
                return False

        # Fallback to AppleScript only if Quartz was unavailable or failed to type
        success = self._unlock_with_applescript(skip_wake=False)

        if success:
            if self._verify_unlock_success(timeout=2.0):
                self.current_state = LockState.UNLOCKED
                return True

        return False


# =============================================================================
# Main Application
# =============================================================================

class CSProximityUnlock:
    """Main application coordinating serial reading and Mac control."""

    def __init__(self, serial_port: str, baud_rate: int = 115200,
                 enable_lock: bool = True, enable_unlock: bool = True):
        self.serial_port = serial_port
        self.baud_rate = baud_rate
        self.enable_lock = enable_lock
        self.enable_unlock = enable_unlock

        self.parser = CSSerialParser()
        self.processor = DistanceProcessor()
        self.mac_controller = MacOSController()

        self.running = False
        self.connected_device: Optional[str] = None
    
    def run(self):
        """Main run loop."""
        # Only require password if unlock is enabled
        if self.enable_unlock and not self.mac_controller.load_password():
            UI.error("No password configured. Run with --setup first.")
            sys.exit(1)

        # Determine mode string
        if self.enable_lock and self.enable_unlock:
            mode_str = "Lock + Unlock"
        elif self.enable_lock:
            mode_str = "Lock only"
        elif self.enable_unlock:
            mode_str = "Unlock only"
        else:
            mode_str = "Monitor only"

        # Show elegant banner
        UI.banner(mode_str, self.serial_port, UNLOCK_DISTANCE_MM, LOCK_DISTANCE_MM)

        try:
            with serial.Serial(self.serial_port, self.baud_rate, timeout=1) as ser:
                UI.info(f"Serial port opened")
                self.running = True
                self._process_serial(ser)

        except serial.SerialException as e:
            UI.error(f"Serial error: {e}")
            sys.exit(1)
        except KeyboardInterrupt:
            UI.clear_line()
            print()
            UI.info("Shutting down...")
    
    def _process_serial(self, ser: serial.Serial):
        """Process serial data stream."""
        while self.running:
            try:
                line = ser.readline().decode('utf-8', errors='ignore').strip()
                if not line:
                    # Check for measurement timeout
                    if self.processor.check_timeout() and self.connected_device:
                        UI.timeout_warning()
                        self._handle_action('lock')
                        self.processor.reset()
                    continue

                # Check for connection/disconnection events
                new_device = self.parser.is_connection_event(line)
                if new_device:
                    UI.device_connected(new_device)
                    self.connected_device = new_device
                    self.processor.reset()
                    continue

                if self.parser.is_disconnection_event(line):
                    UI.device_disconnected()
                    self.connected_device = None
                    self.processor.reset()
                    self._handle_action('lock')
                    continue

                # Check for SOC reset
                if self.parser.is_reset_event(line):
                    UI.soc_reset()
                    self.connected_device = None
                    self.processor.reset()
                    self._handle_action('lock')
                    continue

                # Parse measurement
                measurement = self.parser.parse_line(line)
                if measurement:
                    self._handle_measurement(measurement)

            except Exception as e:
                UI.error(f"Processing error: {e}")
    
    def _handle_measurement(self, m: CSMeasurement):
        """Handle a new measurement."""
        smoothed = self.processor.get_smoothed_distance()

        # Display elegant distance bar
        UI.distance_bar(
            distance_mm=m.distance_mm,
            smoothed_mm=smoothed,
            likelihood=m.likelihood,
            unlock_threshold=UNLOCK_DISTANCE_MM,
            lock_threshold=LOCK_DISTANCE_MM
        )

        # Process and check for action
        action = self.processor.add_measurement(m)
        if action:
            self._handle_action(action)

    def _handle_action(self, action: str):
        """Handle lock/unlock action."""
        if action == 'unlock':
            if not self.enable_unlock:
                return
            if self.mac_controller.current_state != LockState.UNLOCKED:
                success = self.mac_controller.unlock_screen()
                UI.unlock_event(success)

        elif action == 'lock':
            if not self.enable_lock:
                return
            if self.mac_controller.current_state != LockState.LOCKED:
                success = self.mac_controller.lock_screen()
                UI.lock_event(success)


# =============================================================================
# CLI Interface
# =============================================================================

def setup_password():
    """Interactive password setup."""
    import getpass
    c = Colors

    print()
    print(f"{c.CYAN}{c.BOLD}{'─' * 50}{c.RESET}")
    print(f"{c.CYAN}{c.BOLD}  {UI.ICON_BLUETOOTH}  Password Setup{c.RESET}")
    print(f"{c.CYAN}{'─' * 50}{c.RESET}")
    print()
    print(f"  {c.DIM}This will store your Mac password securely in the{c.RESET}")
    print(f"  {c.DIM}macOS Keychain for automatic unlock.{c.RESET}")
    print()
    print(f"  {c.YELLOW}{UI.ICON_WARN} Password is stored encrypted but typed{c.RESET}")
    print(f"  {c.YELLOW}  programmatically. Use for demos only.{c.RESET}")
    print()

    password = getpass.getpass(f"  {c.WHITE}Enter Mac password: {c.RESET}")
    confirm = getpass.getpass(f"  {c.WHITE}Confirm password:   {c.RESET}")

    if password != confirm:
        UI.error("Passwords don't match")
        sys.exit(1)

    if MacOSController.store_password(password):
        print()
        UI.success("Setup complete!")
        print()
        print(f"  {c.DIM}Next steps:{c.RESET}")
        print(f"  {c.WHITE}1.{c.RESET} Grant Accessibility permissions to Terminal/IDE")
        print(f"     {c.DIM}System Preferences → Privacy & Security → Accessibility{c.RESET}")
        print(f"  {c.WHITE}2.{c.RESET} Run: {c.CYAN}python cs_proximity_unlock.py /dev/cu.usbmodem*{c.RESET}")
        print()
    else:
        UI.error("Setup failed")
        sys.exit(1)


def list_serial_ports():
    """List available serial ports."""
    import serial.tools.list_ports
    c = Colors

    ports = serial.tools.list_ports.comports()
    if not ports:
        UI.info("No serial ports found")
        return

    print()
    print(f"  {c.WHITE}{c.BOLD}Available Serial Ports{c.RESET}")
    print(f"  {c.DIM}{'─' * 40}{c.RESET}")
    for port in ports:
        print(f"  {c.CYAN}●{c.RESET} {c.WHITE}{port.device}{c.RESET}")
        if port.description and port.description != 'n/a':
            print(f"    {c.DIM}{port.description}{c.RESET}")
        if port.manufacturer:
            print(f"    {c.DIM}by {port.manufacturer}{c.RESET}")
    print()


def main():
    parser = argparse.ArgumentParser(
        description="Channel Sounding Proximity-Based Mac Lock/Unlock Demo",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s --setup                              Store password in keychain
  %(prog)s --list                               List available serial ports
  %(prog)s /dev/cu.usbmodem14101                Start monitoring (lock + unlock)
  %(prog)s /dev/cu.usbmodem14101 --lock-only    Lock only (no auto-unlock)
  %(prog)s /dev/cu.usbmodem14101 --unlock-only  Unlock only (no auto-lock)
  %(prog)s --delete-password                    Remove stored password

Thresholds can be customized by editing the constants at the top of the script.
        """
    )
    
    parser.add_argument('port', nargs='?', help='Serial port (e.g., /dev/cu.usbmodem14101)')
    parser.add_argument('--setup', action='store_true', help='Configure password storage')
    parser.add_argument('--list', action='store_true', help='List available serial ports')
    parser.add_argument('--delete-password', action='store_true', help='Delete stored password')
    parser.add_argument('--baud', type=int, default=115200, help='Baud rate (default: 115200)')
    parser.add_argument('--lock-only', action='store_true', help='Only enable lock (disable unlock)')
    parser.add_argument('--unlock-only', action='store_true', help='Only enable unlock (disable lock)')
    parser.add_argument('--test-lock', action='store_true', help='Test lock function')
    parser.add_argument('--test-unlock', action='store_true', help='Test unlock function')
    
    args = parser.parse_args()
    
    if args.setup:
        setup_password()
        return
    
    if args.list:
        list_serial_ports()
        return
    
    if args.delete_password:
        MacOSController.delete_password()
        return
    
    if args.test_lock:
        controller = MacOSController()
        controller.lock_screen()
        return
    
    if args.test_unlock:
        controller = MacOSController()
        if controller.load_password():
            controller.unlock_screen()
        else:
            UI.error("No password configured. Run with --setup first.")
        return

    if not args.port:
        parser.print_help()
        print()
        UI.error("No serial port specified. Use --list to see available ports.")
        sys.exit(1)

    # Determine lock/unlock mode
    if args.lock_only and args.unlock_only:
        UI.error("Cannot use both --lock-only and --unlock-only")
        sys.exit(1)

    enable_lock = not args.unlock_only
    enable_unlock = not args.lock_only

    app = CSProximityUnlock(args.port, args.baud, enable_lock, enable_unlock)
    app.run()


if __name__ == "__main__":
    main()
